﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Threading;


namespace ICNC3_Library
{


    partial class ModbusMaster
    {


        #region Enumerations for ICNC3
        /// <summary>
        /// CMD command code codes
        /// </summary>
        public enum DirectCMD
        {
            Clear_alarm = 88,
            Resume_feed_hold = 129,
            Feed_hold = 130,
            Stop_jog_mouvement = 133,
            Reset_speed_override = 138,
            Reset_rapide_speed_override = 143,   //in case of GCODE commande use
            Reset_spindle_speed_override = 153,  //in case of GCODE commande use
            Stop_and_flush = 255
        }





        #endregion
        #region COILS Bits
        const ushort REG_COIL_START = 0;
        const ushort MB_COIL_ADDR_OUTPUT0 = REG_COIL_START + 0;
        #endregion

        #region HOLDING REGISTERS

        // *****************
        // HOLDING REGISTERS
        // *****************

        const ushort REG_HOLDING_START = 2000;	// Adresse de base des Holdings registers
        const ushort MB_HOLD_ADDR_AOUT0 = REG_HOLDING_START + 150;
        const ushort MB_HOLD_ADDR_CMD_BUFFER0 = (REG_HOLDING_START + 0);
        const ushort MB_HOLD_ADDR_CMD_BUFFER149 = (MB_HOLD_ADDR_CMD_BUFFER0 + 149);


        const int ICNC_MB_CMD_STOP_AXE_ID = 100;
        const int ICNC_MB_CMD_STOP_AXES = 101;
        const int ICNC_MB_CMD_MOVE_SPEED = 102;
        const int ICNC_MB_CMD_MOVE_AXE = 103;
        const int ICNC_MB_CMD_MOVE_AXE_RELATIF = 104;
        const int ICNC_MB_CMD_SET_POSITION = 105;
        const int ICNC_MB_CMD_HOME = 106;
        const int ICNC_MB_CMD_PROBE = 107;
        const int ICNC_MB_CMD_PROBE_MASK = 108;
        const int ICNC_MB_CMD_PUT_PLCBASIC_CMD = 200;


        const int ICNC_GRBL_GCODE_STRING__CMD = 1000;
        const int ICNC_GRBL_SET_FEEDRATE_CMD = 1001;	    // Définition de la vitesse d'usinage en mm/mn
        const int ICNC_GRBL_MOVETO_CMD = 1002;              // Commande de déplacement avec spécification des axes à déplacer et de la vitesse à utiliser (feedrate ou rapid)
        const int ICNC_GRBL_MOVE_CIRCULAR_IJ_CMD = 1003;
        const int ICNC_GRBL_BUFF_ACTION_CMD = 1010;
        const int ICNC_GRBL_DWELL_CMD = 1011;
        const int ICNC_WAIT_INPUT_BUFCMD = 1012;
        const int ICNC_GRBL_SET_OVERRIDE_CMD = 1100;
        const int ICNC_GRBL_GO_HOME = 1110;
        const int ICNC_GRBL_DIRECT_CMD = 1200;
        #endregion

        #region Const de test
        const int MAX_AXES = 6;
        const int MIN_ACCEL_DECEL = 1;
        const int MAX_ACCEL_DECEL = 1000000;
        const int MAX_SPEED = 1000000;
        const int MAX_INPUT_PORT = 32;
        const int MAX_OUTPUT_PORT = 12;
        const int MAX_DAC = 8;
        const int FAST_MOVE = (1 << 6);
        #endregion

        #region Global variables

        private byte _unit_id = 1;

        #endregion

        /// <summary>
        /// Get or set ICNC3  Unit ID (similar to modbus slave id)
        /// </summary>
        public byte ICNC3_unit_id
        {
            get { return _unit_id; }
            set { _unit_id = value; }
        }

        #region Testes des variables
        bool RangeCheck(int value, int min, int max)
        {
            if (value > max)
                return false;
            if (value < min)
                return false;
            return true;

        }

        bool is_axeValid(int axeID)
        {
            return RangeCheck(axeID, 1, MAX_AXES);
        }


        bool IS_ACCEL_VALID(int accel)
        {
            return RangeCheck(accel, MIN_ACCEL_DECEL, MAX_ACCEL_DECEL);
        }

        bool IS_DECCEL_VALID(int deccel)
        {
            return RangeCheck(deccel, MIN_ACCEL_DECEL, MAX_ACCEL_DECEL);
        }

        bool IS_SPEED_VALID(int speed)
        {
            return RangeCheck(speed, -MAX_SPEED, MAX_SPEED);
        }
        bool IS_INPUT_VALID(int DINNumber)
        {
            return RangeCheck(DINNumber, 0, MAX_INPUT_PORT * 8 - 1);
        }
        bool IS_OUTPUT_VALID(int DOUTNumber)
        {
            return RangeCheck(DOUTNumber, 0, MAX_OUTPUT_PORT * 8 - 1);
        }
        bool IS_ANALOG_OUTPUT_VALID(int AOUTNumber)
        {
            return RangeCheck(AOUTNumber, 0, MAX_DAC * 8 - 1);
        }
        bool IS_PARAMETERID_VALID(int ParameterNumber)
        {
            return RangeCheck(ParameterNumber, 1, 1999);
        }

        #endregion




        /// <summary>
        /// Read a set of input registers using default ICNC3_unit_id
        /// </summary>
        /// <param name="start_address">Address of first register to be read</param>
        /// <param name="len">Number of registers to be read</param>
        /// <returns>Array of readed registers</returns>
        public ushort[] ReadInputRegisters(ushort start_address, ushort len)
        {
            return ReadInputRegisters(ICNC3_unit_id, start_address, len);
        }



        #region Non bufferized Axes related commands

        /// <summary>
        /// Stop one Axe
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <param name="axeID"></param> from 1 to 6
        /// <returns></returns>number of regesters sent
        public int ICNC3_StopAxeID(int axeID)
        {
            ushort[] data = new ushort[2];
            int res;
            if (!is_axeValid(axeID))
                return -1;


            data[0] = ICNC_MB_CMD_STOP_AXE_ID;
            data[1] = (ushort)axeID;


            try
            {
                WriteMultipleRegisters(ICNC3_unit_id, MB_HOLD_ADDR_CMD_BUFFER0, data);
                res = 2;

            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                res = -1;

            }
            return res;
        }


        /// <summary>
        /// Stop All Axes
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <param name="axeID"></param> axeID = 0x01 to 0x3F
        /// <returns></returns>number of regesters sent
        public int ICNC3_StopAxes(int axeID)
        {

            ushort[] data = new ushort[2];
            int res;


            data[0] = ICNC_MB_CMD_STOP_AXES;
            data[1] = (ushort)axeID;
            try
            {
                WriteMultipleRegisters(ICNC3_unit_id, MB_HOLD_ADDR_CMD_BUFFER0, data);
                res = 2;

            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                res = -1;

            }
            return res;
        }

        /// <summary>
        /// Move one axe to a target
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <param name="AxeID"></param> 1 to 6
        /// <param name="AccelHz"></param>
        /// <param name="SpeedHz"></param>
        /// <param name="DecelHz"></param>
        /// <param name="TargetStep"></param> 
        /// <returns></returns>number of regesters sent
        public int ICNC3_MoveAxeAsync(int AxeID, int AccelHz, int SpeedHz, int DecelHz, int TargetStep)
        {

            int res;

            if (!is_axeValid(AxeID))
                return -1;
            if (!IS_ACCEL_VALID(AccelHz))
                return -1;
            if (!IS_DECCEL_VALID(DecelHz))
                return -1;
            if (!IS_SPEED_VALID(SpeedHz))
                return -1;


            ushort[] data = new ushort[10];
            ushort[] two_int = new ushort[2];


            data[0] = ICNC_MB_CMD_MOVE_AXE;
            data[1] = (ushort)AxeID;
            two_int = ConvertIntToRegisters(AccelHz);
            data[2] = two_int[0];
            data[3] = two_int[1];
            two_int = ConvertIntToRegisters(SpeedHz);
            data[4] = two_int[0];
            data[5] = two_int[1];
            two_int = ConvertIntToRegisters(DecelHz);
            data[6] = two_int[0];
            data[7] = two_int[1];
            two_int = ConvertIntToRegisters(TargetStep);
            data[8] = two_int[0];
            data[9] = two_int[1];

            try
            {
                WriteMultipleRegisters(ICNC3_unit_id, REG_HOLDING_START, data);

                res = 10;
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                res = -1;
            }

            return res;

        }

        /// <summary>
        /// move one axe with ginen speed
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <param name="axeID"></param> 1 to 6
        /// <param name="accel_decel"></param> acceleration and decceleration value
        /// <param name="speed"></param>
        /// <returns></returns>number of regesters sent
        public int ICNC3_MoveSpeed(int axeID, int accel_decel, int speed)
        {
            int res;
            ushort[] data = new ushort[6];
            ushort[] two_int = new ushort[2];


            if (!is_axeValid(axeID))
                return -1;
            if (!IS_ACCEL_VALID(accel_decel))
                return -1;
            if (!IS_SPEED_VALID(speed))
                return -1;


            data[0] = ICNC_MB_CMD_MOVE_SPEED;
            data[1] = (ushort)axeID;
            two_int = ConvertIntToRegisters(accel_decel);
            data[2] = two_int[0];
            data[3] = two_int[1];
            two_int = ConvertIntToRegisters(speed);
            data[4] = two_int[0];
            data[5] = two_int[1];


            try
            {
                WriteMultipleRegisters(ICNC3_unit_id, MB_HOLD_ADDR_CMD_BUFFER0, data);
                res = 6;

            }
            catch (Exception ex)
            {
                res = -1;
                Console.WriteLine(ex.Message);
            }

            return res;
        }

        /// <summary>
        /// Homing
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <param name="AxeID"></param> 1 to 6
        /// <param name="HomingMode"></param>  HomingMode not used at this time
        /// <param name="InputNumber"></param>
        /// <param name="ExceptedInputState"></param> 0 if DIN=0 when switch is pressed (NC contact type)
        /// <param name="Direction"></param>  >0 for positive direction, <=0 for neative direction
        /// <param name="HighSpeed"></param>
        /// <param name="LowSpeed"></param>
        /// <param name="Acceleration"></param> acceleration
        /// <param name="Deceleration"></param>decceleration
        /// <param name="StrokeLimitStep"></param>
        /// <param name="ReverseTimer"></param>
        /// <param name="HomePositionStep"></param>
        /// <param name="ClearanceStep"></param>
        /// <returns></returns>number of regesters sent
        public int ICNC3_HomeAxe(
    int AxeID,
    int HomingMode,
    int InputNumber,
    int ExceptedInputState,
    int Direction,
    int HighSpeed,
    int LowSpeed,
    int Acceleration,
    int Deceleration,
    int StrokeLimitStep,
    int ReverseTimer,
    int HomePositionStep,
    int ClearanceStep)
        {
            int res;
            if (!is_axeValid(AxeID))
                return -1;
            if (!IS_ACCEL_VALID(Acceleration))
                return -1;
            if (!IS_DECCEL_VALID(Deceleration))
                return -1;
            if (!IS_SPEED_VALID(HighSpeed))
                return -1;
            if (!IS_SPEED_VALID(LowSpeed))
                return -1;
            ushort[] data = new ushort[21];
            ushort[] two_int = new ushort[2];









            data[0] = ICNC_MB_CMD_HOME;
            data[1] = (ushort)AxeID;
            data[2] = (ushort)HomingMode;
            data[3] = (ushort)InputNumber;
            data[4] = (ushort)ExceptedInputState;
            data[5] = (ushort)Direction;
            two_int = ConvertIntToRegisters(HighSpeed);
            data[6] = two_int[0];
            data[7] = two_int[1];
            two_int = ConvertIntToRegisters(LowSpeed);
            data[8] = two_int[0];
            data[9] = two_int[1];
            two_int = ConvertIntToRegisters(Acceleration);
            data[10] = two_int[0];
            data[11] = two_int[1];
            two_int = ConvertIntToRegisters(Deceleration);
            data[12] = two_int[0];
            data[13] = two_int[1];
            two_int = ConvertIntToRegisters(StrokeLimitStep);
            data[14] = two_int[0];
            data[15] = two_int[1];
            data[16] = (ushort)ReverseTimer;
            two_int = ConvertIntToRegisters(HomePositionStep);
            data[17] = two_int[0];
            data[18] = two_int[1];
            two_int = ConvertIntToRegisters(ClearanceStep);
            data[19] = two_int[0];
            data[20] = two_int[1];



            try
            {
                WriteMultipleRegisters(ICNC3_unit_id, REG_HOLDING_START, data);
                res = 21;

            }
            catch (Exception ex)
            {
                res = -1;
                Console.WriteLine(ex.Message);
            }

            return res;
        }



        /// <summary>
        /// 
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <param name="AxeID"></param> 1 to 6
        /// <param name="InputNumber"></param>
        /// <param name="ExceptedInputState"></param> 0 if DIN=0 when switch is pressed (NC contact type)
        /// <param name="Acceleration"></param>
        /// <param name="Speed"></param>
        /// <param name="Deceleration"></param>
        /// <param name="StrokeLimitStep"></param> StrokeLimitStep is positive for positive direction, negative for negative direction
        /// <returns></returns>number of regesters sent
        public int ICNC3_ProbeAxe(
    int AxeID,
    int InputNumber,
    int ExceptedInputState,
    int Acceleration,
    int Speed,
    int Deceleration,
    int StrokeLimitStep)
        {
            int res;
            if (!is_axeValid(AxeID))
                return -1;

            if (!IS_INPUT_VALID(InputNumber))
                return -1;
            if (!IS_DECCEL_VALID(Deceleration))
                return -1;
            if (!IS_ACCEL_VALID(Acceleration))
                return -1;
            if (!IS_SPEED_VALID(Speed))
                return -1;
            ushort[] data = new ushort[12];
            ushort[] two_int = new ushort[2];




            data[0] = ICNC_MB_CMD_PROBE;
            data[1] = (ushort)AxeID;
            data[2] = (ushort)InputNumber;
            data[3] = (ushort)ExceptedInputState;
            two_int = ConvertIntToRegisters(Acceleration);
            data[4] = two_int[0];
            data[5] = two_int[1];
            two_int = ConvertIntToRegisters(Speed);
            data[6] = two_int[0];
            data[7] = two_int[1];
            two_int = ConvertIntToRegisters(Deceleration);
            data[8] = two_int[0];
            data[9] = two_int[1];
            two_int = ConvertIntToRegisters(StrokeLimitStep);
            data[10] = two_int[0];
            data[11] = two_int[1];
            try
            {
                WriteMultipleRegisters(ICNC3_unit_id, REG_HOLDING_START, data);
                res = 12;

            }
            catch (Exception ex)
            {
                res = -1;
                Console.WriteLine(ex.Message);
            }

            return res;
        }

        /// <summary>
        /// Probe Axe With Mask
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <param name="AxeID"></param> 1 to 6
        /// <param name="ANDMask"></param>
        /// <param name="XORMask"></param>
        /// <param name="Acceleration"></param>
        /// <param name="Speed"></param>
        /// <param name="Deceleration"></param>
        /// <param name="StrokeLimitStep"></param>
        /// <returns></returns>number of regesters sent
        public int ICNC3_ProbeAxeWithMask(
     int AxeID,
     int ANDMask,
     int XORMask,
     int Acceleration,
     int Speed,
     int Deceleration,
     int StrokeLimitStep)
        {
            int res;
            ushort[] data = new ushort[14];
            ushort[] two_int = new ushort[2];

            if (!is_axeValid(AxeID))
                return -1;
            if (!IS_DECCEL_VALID(Deceleration))
                return -1;
            if (!IS_ACCEL_VALID(Acceleration))
                return -1;
            if (!IS_SPEED_VALID(Speed))
                return -1;



            data[0] = ICNC_MB_CMD_PROBE_MASK;
            data[1] = (ushort)AxeID;
            two_int = ConvertIntToRegisters(ANDMask);
            data[2] = two_int[0];
            data[3] = two_int[1];
            two_int = ConvertIntToRegisters(XORMask);
            data[4] = two_int[0];
            data[5] = two_int[1];
            two_int = ConvertIntToRegisters(Acceleration);
            data[6] = two_int[0];
            data[7] = two_int[1];
            two_int = ConvertIntToRegisters(Speed);
            data[8] = two_int[0];
            data[9] = two_int[1];
            two_int = ConvertIntToRegisters(Deceleration);
            data[10] = two_int[0];
            data[11] = two_int[1];
            two_int = ConvertIntToRegisters(StrokeLimitStep);
            data[12] = two_int[0];
            data[13] = two_int[1];




            try
            {
                WriteMultipleRegisters(ICNC3_unit_id, REG_HOLDING_START, data);
                res = 10;

            }
            catch (Exception ex)
            {
                res = -1;
                Console.WriteLine(ex.Message);
            }

            return res;
        }



        /// <summary>
        /// Direct write on axes position register
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <param name="AxeID"></param> 1 to 6
        /// <param name="StepPosition"></param>
        /// <returns></returns>number of regesters sent
        public int ICNC3_SetPosition(int AxeID, int StepPosition)
        {
            int res;
            ushort[] data = new ushort[4];
            ushort[] two_int = new ushort[2];

            if (!is_axeValid(AxeID))
                return -1;


            data[0] = ICNC_MB_CMD_SET_POSITION;
            data[1] = (ushort)AxeID;
            two_int = ConvertIntToRegisters(StepPosition);
            data[2] = two_int[0];
            data[3] = two_int[1];


            try
            {
                WriteMultipleRegisters(ICNC3_unit_id, REG_HOLDING_START, data);
                res = 10;

            }
            catch (Exception ex)
            {
                res = -1;
                Console.WriteLine(ex.Message);
            }

            return res;
        }

        #endregion

        #region Non bufferized IO related commands

        /// <summary>
        /// Set Digital output state
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <param name="OutputNumber"></param> OutputNumbre from 0 to MAX_OUTPUT_PORT*8-1
        /// <param name="value"></param>Value : 0 for reset, other value for set
        /// <returns></returns>number of regesters sent
        public int ICNC3_SetOutput(int OutputNumber, bool value)
        {

            int res;
            if (!IS_OUTPUT_VALID(OutputNumber))
                return -1;

            
            try
            {
                WriteSingleCoil(ICNC3_unit_id, (ushort)(MB_COIL_ADDR_OUTPUT0 + OutputNumber), value);

                res = 1;

            }
            catch (Exception ex)
            {

                res = -1;
                Console.WriteLine(ex.Message);
            }
           
            return res;

        }

        /// <summary>
        /// Set Analog Output output Value
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <param name="AnalogOutputNumber"></param> 0 or 1
        /// <param name="value"></param>
        /// <returns></returns>number of regesters sent
        public int ICNC3_SetAnalogOutput( int AnalogOutputNumber, int value)
        {
            int res;
            if (!IS_ANALOG_OUTPUT_VALID(AnalogOutputNumber))
                return -1;

           
            try
            {
                WriteSingleRegister(ICNC3_unit_id,(ushort) (MB_HOLD_ADDR_AOUT0 + AnalogOutputNumber), (ushort)value);
                res = 1;

            }
            catch (Exception ex)
            {
                res = -1;
                Console.WriteLine(ex.Message);
            }
               
            return res;

        }

        /// <summary>
        /// Write a single setings regesters 
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <param name="ParameterNumber"></param> from 1 to 1999
        /// <param name="ivalue"></param>
        /// <returns></returns>number of regesters sent
        public int ICNC3_SetParameter32bits( int ParameterNumber, int ivalue)
        {
            int res;
            ushort[] data = new ushort[2];
            ushort[] two_int = new ushort[2];


            if (!IS_PARAMETERID_VALID(ParameterNumber))
                return -1;

            

            two_int = ConvertIntToRegisters(ivalue);
            data[0] = two_int[0];
            data[1] = two_int[1];

            try
            {
               WriteMultipleRegisters(ICNC3_unit_id ,(ushort)(32000 + (ParameterNumber * 2)), data);
                res = 2;

            }
            catch (Exception ex)
            {
                res = -1;
                Console.WriteLine(ex.Message);
            }
            
            return res;

        }

        /// <summary>
        /// Read a single setings regesters 
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <param name="ParameterNumber"></param> from 1 to 1999
        /// <param name="ivalue"></param>
        /// <returns></returns>number of regesters sent
        public int ICNC3_GetParameter32bits(int ParameterNumber, out int ivalue)
        {
            int res = -1;
            ushort[] two_int = new ushort[2];
            int val;

            ivalue = 0;


            if (!IS_PARAMETERID_VALID(ParameterNumber))
                return -1;
            try
            {

                two_int = ReadHoldingRegisters(ICNC3_unit_id, (ushort)(32000 + (ParameterNumber * 2)), 2);

                val = ConvertRegistersToInt(two_int);

                ivalue = val;


                res = 1;

            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                res = -1;

            }
            return res;

        }

        /// <summary>
        /// Write a single setings regesters (Float value)
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <param name="ParameterNumber"></param> from 1 to 1999
        /// <param name="fvalue"></param>
        /// <returns></returns>number of regesters sent
        public int ICNC3_SetParameterFloat( int ParameterNumber, float fvalue)
        {

            int res;
            ushort[] two_int = new ushort[2];


            if (!IS_PARAMETERID_VALID(ParameterNumber))
                return -1;

           

            two_int = ConvertFloatToRegisters(fvalue);


            try
            {
                WriteMultipleRegisters( ICNC3_unit_id,(ushort)(32000 + (ParameterNumber * 2)), two_int);
                res = 2;

            }
            catch (Exception ex)
            {
                res = -1;
                Console.WriteLine(ex.Message);
            }
            
            return res;

        }

        /// <summary>
        /// Read a single setings regesters (Float value)
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <param name="ParameterNumber"></param> from 1 to 1999
        /// <param name="fvalue"></param>
        /// <returns></returns>number of regesters sent
        public int ICNC3_GetParameterFloat( int ParameterNumber, out float fvalue)
        {
            int res;
            ushort[] data = new ushort[2];
            ushort[] two_int = new ushort[2];
            float val;
            fvalue = 0;
            if (!IS_PARAMETERID_VALID(ParameterNumber))
                return -1;
      
            try
            {
                two_int = ReadHoldingRegisters(ICNC3_unit_id,(ushort)( 32000 + (ParameterNumber * 2)), 2);
                val = ConvertRegistersToFloat(two_int);
                fvalue = val;


                res = 1;

            }
            catch (Exception ex)
            {
                res = -1;
                Console.WriteLine(ex.Message);
            }
           
            return res;

        }

        #endregion

        #region Interact command with CNC process
        /// <summary>
        /// Send a direct non-bufferised command
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <param name="Command"></param>
        public int ICNC3_CNCDirectCommand( int Command)
        {
            int res;
            
            ushort[] data = new ushort[2];

            data[0] = ICNC_GRBL_DIRECT_CMD;
            data[1] = (ushort)Command;

            try
            {
                WriteMultipleRegisters(ICNC3_unit_id,MB_HOLD_ADDR_CMD_BUFFER0, data);
                res = 2;

            }
            catch (Exception ex)
            {
                res = -1;
                Console.WriteLine(ex.Message);
            }
          
            return res;
        }


        /// <summary>
        /// Define override value( Value are pourcentage of speed setting)
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <param name="NormalOverride_pct"></param> NormalOverride_pct is related to the feed override previously define with ICNC3_SetFeedRate command
        /// <param name="RapidOverride_pct"></param>RapidOverride_pct is related to the maximum speed
        /// <returns></returns>number of regesters sent
        public int CNC3_CNCSetOverride( int NormalOverride_pct, int RapidOverride_pct)
        {
            int res;

            ushort[] data = new ushort[3];
            data[0] = ICNC_GRBL_SET_OVERRIDE_CMD;
            data[1] = (ushort)NormalOverride_pct;
            data[2] = (ushort)RapidOverride_pct;
            try
            {
                WriteMultipleRegisters(ICNC3_unit_id,MB_HOLD_ADDR_CMD_BUFFER0, data);
                res = 3;

            }
            catch (Exception ex)
            {
                res = -1;
                Console.WriteLine(ex.Message);
            }
            
            return res;

        }


        /// <summary>
        /// Start automatic homing sequence difine by parameters
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <param name="AxeID"></param>AxeID = 0 to 6 / if AxeID=0, start the full sequence
        /// For other value, launch individual axis sequence
        /// <returns></returns>number of regesters sent
        public int ICNC3_StartHomingSequence(int AxeID)
        {
            int res;

            ushort[] data = new ushort[3];

            data[0] = ICNC_GRBL_GO_HOME;
            data[1] = (ushort)(AxeID & 0x3f);
            try
            {
               WriteMultipleRegisters(ICNC3_unit_id,MB_HOLD_ADDR_CMD_BUFFER0, data);
                res = 10;

            }
            catch (Exception ex)
            {
                res = -1;
                Console.WriteLine(ex.Message);
            }
            return res;

        }
        #endregion

        #region Commandes CNC bufferisées

        /// <summary>
        /// Send a Gcode line directly
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <param name="GCodeStr"></param> Gcode line (string) separator signe "."
        /// <returns></returns>number of regesters sent
        public int ICNC3_PushGCode( string GCodeStr)
        {
            int res;
            
            int registerRequired;
            registerRequired = (GCodeStr.Length + 1) / 2;
            ushort[] data = new ushort[registerRequired + 2];
            int[] strigg = new int[registerRequired];

            data[0] = ICNC_GRBL_GCODE_STRING__CMD;
            data[1] = (ushort)GCodeStr.Length;
            strigg = ConvertStringToRegisters(GCodeStr);
            for (int i = 0; i < registerRequired; i++)
            {
                data[i + 2] = (ushort)strigg[i];
            }


            try
            {
               WriteMultipleRegisters(ICNC3_unit_id, MB_HOLD_ADDR_CMD_BUFFER0, data);
                res = registerRequired + 2;

            }
            catch (Exception ex)
            {
                res = -1;
                Console.WriteLine(ex.Message);
            }
           
            return res;
        }
        /// <summary>
        /// Send a Command(string) it need to end by "\n"
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <param name="strCommand"></param> Command (string) ex:"out 1,1\n"
        /// <returns></returns>number of regesters sent
        public int ICNC3_SendPLCCommand(string strCommand)
        {
            int res;
            
            int registerRequired;
            registerRequired = (strCommand.Length + 1) / 2;
            ushort[] data = new ushort[registerRequired + 2];
            int[] strigg = new int[registerRequired];

            data[0] = ICNC_MB_CMD_PUT_PLCBASIC_CMD;
            data[1] = (ushort)strCommand.Length;
            strigg = ConvertStringToRegisters(strCommand);
            for (int i = 0; i < registerRequired; i++)
            {
                data[i + 2] = (ushort)strigg[i];

            }
            try
            {
                WriteMultipleRegisters(ICNC3_unit_id, MB_HOLD_ADDR_CMD_BUFFER0, data);
                res = registerRequired + 2;

            }
            catch (Exception ex)
            {
                res = -1;
                Console.WriteLine(ex.Message);
            }
            
            return res;
        }



        /// <summary>
        /// Adjust feed rate for following bufferized motion
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <param name="feed_rate_mm_per_mn"></param>
        /// <returns></returns>number of regesters sent
        public int ICNC3_PushSetFeedRate(float feed_rate_mm_per_mn)
        {
            int res;

            ushort[] data = new ushort[3];
            ushort[] two_int = new ushort[2];

            data[0] = ICNC_GRBL_SET_FEEDRATE_CMD;
            two_int = ConvertFloatToRegisters(feed_rate_mm_per_mn);
            data[1] = two_int[0];
            data[2] = two_int[1];
            /* Envoie dans le buffer de commandes */
            try
            {
                WriteMultipleRegisters(ICNC3_unit_id ,MB_HOLD_ADDR_CMD_BUFFER0, data);
                res = 3;

            }
            catch (Exception ex)
            {
                res = -1;
                Console.WriteLine(ex.Message);
            }
           
            return res;
        }

        /// <summary>
        /// Send a movement 
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <param name="AxesFlag"></param> axe1 = lsb 
        /// <param name="RapidMove"></param>
        /// <param name="x"></param>
        /// <param name="y"></param>
        /// <param name="z"></param>
        /// <param name="a"></param>
        /// <param name="b"></param>
        /// <param name="c"></param>
        /// <returns></returns>number of regesters sent
        public int ICNC3_PushMoveTo( int AxesFlag, float RapidMove, float x, float y, float z, float a, float b, float c)
        {
            int res;

            ushort[] data = new ushort[2 + (MAX_AXES * 2)];
            ushort[] two_int = new ushort[2];
            int index = 2;


            data[0] = ICNC_GRBL_MOVETO_CMD;
            data[1] =(ushort)(AxesFlag & 0x3F);

            if (RapidMove > 0)
            {
                data[1] |= (ushort)FAST_MOVE;

            }
            if ((AxesFlag & (1 << 0)) != 0)
            {
                two_int = ConvertFloatToRegisters(x);
                data[index] = two_int[0];
                data[index + 1] = two_int[1];
                index = index + 2;

            }
            if ((AxesFlag & (1 << 1)) != 0)
            {
                two_int = ConvertFloatToRegisters(y);
                data[index] = two_int[0];
                data[index + 1] = two_int[1];
                index = index + 2;

            }
            if ((AxesFlag & (1 << 2)) != 0)
            {
                two_int =ConvertFloatToRegisters(z);
                data[index] = two_int[0];
                data[index + 1] = two_int[1];
                index = index + 2;

            }
            if ((AxesFlag & (1 << 3)) != 0)
            {
                two_int = ConvertFloatToRegisters(a);
                data[index] = two_int[0];
                data[index + 1] = two_int[1];
                index = index + 2;

            }
            if ((AxesFlag & (1 << 4)) != 0)
            {
                two_int = ConvertFloatToRegisters(b);
                data[index] = two_int[0];
                data[index + 1] = two_int[1];
                index = index + 2;

            }
            if ((AxesFlag & (1 << 5)) != 0)
            {
                two_int = ConvertFloatToRegisters(c);
                data[index] = two_int[0];
                data[index + 1] = two_int[1];
                index = index + 2;

            }

            ushort[] data1 = new ushort[index];
            for (int i = 0; i < index; i++)
            {
                data1[i] = data[i];
            }

            try
            {

                WriteMultipleRegisters(ICNC3_unit_id ,MB_HOLD_ADDR_CMD_BUFFER0, data1);
                res = index;

            }
            catch (Exception ex)
            {
                res = -1;
                Console.WriteLine(ex.Message);
            }
           
            return res;

        }
        /// <summary>
        /// Send circular motion
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <param name="Direction"></param> 0=CW / 1 =CCW
        /// <param name="x"></param>
        /// <param name="y"></param>
        /// <param name="I"></param>
        /// <param name="J"></param>
        /// <returns></returns>number of regesters sent
        public int ICNC3_PushCircleCommon( int Direction, float x, float y, float I, float J)
        {
            int res;

            ushort[] data = new ushort[10];
            ushort[] two_int = new ushort[2];

            data[0] = ICNC_GRBL_MOVE_CIRCULAR_IJ_CMD;
            data[1] = (ushort)Direction;      // 0 for CW direction, 1 for CCW direction
            two_int = ConvertFloatToRegisters(x);
            data[2] = two_int[0];
            data[3] = two_int[1];
            two_int = ConvertFloatToRegisters(y);
            data[4] = two_int[0];
            data[5] = two_int[1];
            two_int = ConvertFloatToRegisters(I);
            data[6] = two_int[0];
            data[7] = two_int[1];
            two_int = ConvertFloatToRegisters(J);
            data[8] = two_int[0];
            data[9] = two_int[1];

            try
            {
                WriteMultipleRegisters(ICNC3_unit_id,MB_HOLD_ADDR_CMD_BUFFER0, data);
                res = 10;

            }
            catch (Exception ex)
            {
                res = -1;
                Console.WriteLine(ex.Message);
            }
            
            return res;
        }

        public int ICNC3_PushCircleCW(float x, float y, float I, float J)
        {

            return ICNC3_PushCircleCommon( 0, x, y, I, J);
        }
        public int ICNC3_PushCircleCCW( float x, float y, float I, float J)
        {
            return ICNC3_PushCircleCommon(1, x, y, I, J);
        }



        /// <summary>
        /// Add delay (ms) into then command buffer
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <param name="delay_ms"></param>
        /// <returns></returns>number of regesters sent
        public int ICNC3_PushDelay( int delay_ms)
        {
            int res;

            ushort[] data = new ushort[2];
            data[0] = ICNC_GRBL_DWELL_CMD;
            data[1] = (ushort)delay_ms;
            try
            {
               WriteMultipleRegisters(ICNC3_unit_id,MB_HOLD_ADDR_CMD_BUFFER0, data);
                res = 10;

            }
            catch (Exception ex)
            {
                res = -1;
                Console.WriteLine(ex.Message);
            }
            
            return res;
        }



        /// <summary>
        /// common function used to push action into modbus buffer
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <param name="ActionType"></param> functions below and Action type 1 for DOUT, 2 for AOUT, 3 for holding register
        /// <param name="Adress"></param>  ex:DOutNumber,AOutNumber,HoldingAddress ...
        /// <param name="Value"></param>
        /// <returns></returns>number of regesters sent
        public int ICNC3_PushActionCommon( int ActionType, int Adress, int Value)
        {
            int res;

            ushort[] data = new ushort[4];
            data[0] = ICNC_GRBL_BUFF_ACTION_CMD;
            data[1] = (ushort)ActionType;
            data[2] = (ushort)Adress;
            data[3] = (ushort)Value;
            try
            {
               WriteMultipleRegisters(ICNC3_unit_id,MB_HOLD_ADDR_CMD_BUFFER0, data);
                res = 4;

            }
            catch (Exception ex)
            {
                res = -1;
                Console.WriteLine(ex.Message);
            }
            
            return res;
        }



        /// <summary>
        /// Add digital output set state in command buffer
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <param name="DOutNumber"></param>
        /// <param name="Value"></param>
        /// <returns></returns>number of regesters sent
        public int ICNC3_PushActionDOUT( int DOutNumber, int Value)
        {
            if (!IS_OUTPUT_VALID(DOutNumber))
                return -1;

            try
            {
                ICNC3_PushActionCommon(1, DOutNumber, Value);
                return 4;

            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                return -1;
                
            }

        }



        /// <summary>
        /// Add analog output set value in command buffer
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <param name="AOutNumber"></param>
        /// <param name="Value"></param>
        /// <returns></returns> number of regesters sent
        public int ICNC3_PushActionAOUT(int AOutNumber, int Value)
        {
            if (!IS_ANALOG_OUTPUT_VALID(AOutNumber))
                return -1;

            try
            {
                ICNC3_PushActionCommon( 2, AOutNumber, Value);
                return 4;
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                return -1;
            }
        }

        /// <summary>
        /// Add Modbus Holding Register set value in command buffer
        /// </summary>
        /// <param name="ctx"></param> Modbus Connection
        /// <param name="HoldingAddress"></param> 
        /// <param name="Value"></param>
        /// <returns></returns> number of regesters sent
        public int ICNC3_PushActionRegister(int HoldingAddress, int Value)
        {
            try
            {
                ICNC3_PushActionCommon( 3, HoldingAddress, Value);
                return 4;
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                return -1;
            }
        }
        /// <summary>
        ///  Add Modbus THC activation with auto tune of target voltage  in command buffer 
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <returns></returns>number of regesters sent
        ///THC function is delayed and start after a timer (parameter 1351, type float, unit second),
        ///Then, THC regulation start with target voltage define in Holding register . 
        ///Target voltage have to be define before but can be change on the fly
        ///Registre 2444, 32 bits, unit mV	
        ///While THC is ON, Z axis is locked and can't be control by external command
        public int ICNC3_PushActionTHCOn()
        {
            try
            {
                ICNC3_PushActionCommon( 4, 0, 0);
                return 4;
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                return -1;
            }
        }
        /// <summary>
        /// Add Modbus THC activation with auto tune of target voltage  in command buffer 
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <param name="SamplingTime"></param>Then sampling and average is done for SamplingTime period
        /// <returns></returns>number of regesters sent
        ///THC function is delayed and startafter a timer(parameter 1351, type float, unit second),
        ///Then, THC regulation start with target voltage previously measured.
        ///Voltage result of measurement is stored into Holding register 2444 (32 bits, unit mV) and can be change on the fly
        ///While THC is ON, Z axis is locked and can't be control by external command

        public int ICNC3_PushActionTHCOnAuto( int SamplingTime)
        {
            try
            {
                ICNC3_PushActionCommon( 5, SamplingTime, 0);
                return 4;
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                return -1;
                
            }
        }

        /// <summary>
        /// Add Modbus THC Off in command buffer
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <returns></returns>number of regesters sent
        public int ICNC3_PushActionTHCOff()
        {
            try
            {
                ICNC3_PushActionCommon( 6, 0, 0);
                return 4;
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                return -1;
            }
        }
        /// <summary>
        /// Add Modbus THC Pause in command buffer
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <returns></returns>number of regesters sent
        /// Pause THC while result of stop Z axis and disable regulation.
        ///It can be use to disable the THC in some portion of trajectory */
        public int ICNC3_PushActionTHCPause()
        {
            try
            {
                ICNC3_PushActionCommon( 7, 0, 0);
                return 4;
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                return -1;
            }
        }

        /// <summary>
        /// Add Modbus THC Resume in command buffer
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <returns></returns>number of regesters sent
        /// Resume THC will re-activate the regulation after a pause
        public int ICNC3_PushActionTHCResume()
        {
            try
            {
                ICNC3_PushActionCommon( 8, 0, 0);
                return 4;
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                return -1;
            }
        }











        /// <summary>
        /// Add wait event on digital input into then command buffer
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <param name="DIN_EventType"></param>  1 for rising, 2 for falling, 3 for high, 4 for low
        /// <param name="DIN_number"></param>
        /// <param name="Timeout_action"></param>
        /// <param name="timeout_value_ms"></param>
        /// <returns></returns>number of regesters sent
        public int ICNC3_PushWaitForDINEvent( int DIN_EventType, int DIN_number, int Timeout_action, int timeout_value_ms)
        {
            int res;

            ushort[] data = new ushort[6];


            data[0] = ICNC_WAIT_INPUT_BUFCMD;   // Wait event modbus command
            data[1] = 1;    // Action type 1 for wait input state
            data[2] = (ushort)timeout_value_ms;
            data[3] = (ushort)Timeout_action;
            data[4] = (ushort)DIN_EventType;
            data[5] = (ushort)DIN_number;


            try
            {
                WriteMultipleRegisters(ICNC3_unit_id,MB_HOLD_ADDR_CMD_BUFFER0, data);
                res = 6;

            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                res = -1;

            }
           
            return res;
        }


        /// <summary>
        /// Add wait event on analog input input into then command buffer
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <param name="AIN_EventType"></param> 1 for higher than, 2 lower than, 3 for equal
        /// <param name="AIN_number"></param>
        /// <param name="AIN_value"></param>
        /// <param name="Timeout_action"></param>
        /// <param name="timeout_value_ms"></param>
        /// <returns></returns>number of regesters sent
        public int ICNC3_PushWaitForAINEvent(int AIN_EventType, int AIN_number, int AIN_value, int Timeout_action, int timeout_value_ms)
        {
            int res;

            ushort[] data = new ushort[7];
            data[0] = ICNC_WAIT_INPUT_BUFCMD;
            data[1] = 2;    // Action type 2 for wait analog input value
            data[2] = (ushort)timeout_value_ms;
            data[3] = (ushort)Timeout_action;
            data[4] = (ushort)AIN_EventType;
            data[5] = (ushort)AIN_value;
            data[6] = (ushort)AIN_number;

            try
            {
               WriteMultipleRegisters(ICNC3_unit_id,MB_HOLD_ADDR_CMD_BUFFER0, data);
                res = 7;

            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                res = -1;

            }
           
            return res;
        }


        /// <summary>
        /// Add wait event on modbus variabme (using extended adress type) into then command buffer
        /// </summary>
        /// <param name="ctx"></param>Modbus Connection
        /// <param name="ModbusData_EventType"></param>1 for higher than, 2 lower than, 3 for equal
        /// <param name="ExtendedModbusAddress"></param>
        /// <param name="ModbusVariableValue"></param>
        /// <param name="Timeout_action"></param>
        /// <param name="timeout_value_ms"></param>
        /// <returns></returns>number of regesters sent
        public int ICNC3_PushWaitForRegisterEvent( int ModbusData_EventType, int ExtendedModbusAddress, int ModbusVariableValue, int Timeout_action, int timeout_value_ms)
        {
            int res;
           
            ushort[] data = new ushort[8];
            ushort[] two_int = new ushort[2];


            data[0] = ICNC_WAIT_INPUT_BUFCMD;
            data[1] = 3;    // Action type 3 for wait extended modbus variable 
            data[2] = (ushort)timeout_value_ms;
            data[3] = (ushort)Timeout_action;
            data[4] = (ushort)ModbusData_EventType;
            data[5] = (ushort)ModbusVariableValue;
            two_int = ConvertDoubleToRegisters(ExtendedModbusAddress);
            data[6] = two_int[0];
            data[7] = two_int[1];
            try
            {
                WriteMultipleRegisters(ICNC3_unit_id,MB_HOLD_ADDR_CMD_BUFFER0, data);
                res = 8;

            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                res = -1;

            }
          

            return res;
        }
        #endregion

 
    }
}
