﻿using Gcodes.Tokens;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text.RegularExpressions;

namespace Gcodes
{
    /// <summary>
    /// A tokenizer for converting a stream of characters into a stream of
    /// <see cref="Token"/>s.
    /// </summary>
    public class Lexer
    {
        List<Pattern> patterns;
        List<Regex> skips;
        string src;
        int pointer;
        int lineNumber;

        internal bool Finished => pointer >= src.Length;

        /// <summary>
        /// Event fired whenever a comment is encountered.
        /// </summary>
        public event EventHandler<CommentEventArgs> CommentDetected;

        /// <summary>
        /// Create a new <see cref="Lexer"/> which will tokenize the provided 
        /// source text.
        /// </summary>
        /// <param name="src"></param>
        public Lexer(string src)
        {
            skips = new List<Regex>
            {
                new Regex(@"\G\s+", RegexOptions.Compiled),
                new Regex(@"\G;([^\n\r]*)", RegexOptions.Compiled),
                new Regex(@"\G\(([^)\n\r]*)\)", RegexOptions.Compiled),
            };
            this.src = src;
            pointer = 0;
            lineNumber = 0;

            string Separator = System.Globalization.CultureInfo.CurrentUICulture.NumberFormat.NumberDecimalSeparator;
            string local_patern;
            if (Separator == ",")
                local_patern = @"[-+]?(\d+\,\d+|\,\d+|\d+\,?)";
            else
                local_patern = @"[-+]?(\d+\.\d+|\.\d+|\d+\.?)";



            patterns = new List<Pattern>
            {
                new Pattern(@"G", TokenKind.G),
                new Pattern(@"O", TokenKind.O),
                new Pattern(@"N", TokenKind.N),
                new Pattern(@"M", TokenKind.M),
                new Pattern(@"T", TokenKind.T),
                new Pattern(@"X", TokenKind.X),
                new Pattern(@"Y", TokenKind.Y),
                new Pattern(@"Z", TokenKind.Z),
                new Pattern(@"F", TokenKind.F),
                new Pattern(@"I", TokenKind.I),
                new Pattern(@"J", TokenKind.J),
                new Pattern(@"K", TokenKind.K),
                new Pattern(@"A", TokenKind.A),
                new Pattern(@"B", TokenKind.B),
                new Pattern(@"C", TokenKind.C),
                new Pattern(@"H", TokenKind.H),
                new Pattern(@"P", TokenKind.P),
                new Pattern(@"S", TokenKind.S),

                new Pattern(local_patern, TokenKind.Number),
                //new Pattern(@"[-+]?(\d+\.\d+|\.\d+|\d+\.?)", TokenKind.Number),

            };
        }

        /// <summary>
        /// Start tokenizing the input.
        /// </summary>
        /// <returns></returns>
        public IEnumerable<Token> Tokenize()
        {
            while (!Finished)
            {
                SkipStuff();
                if (Finished) break;
                yield return NextToken();
            }
        }

        private void SkipStuff()
        {
            int currentPass;

            do
            {
                currentPass = pointer;

                foreach (var skip in skips)
                {
                    var match = skip.Match(src, pointer);

                    if (match.Success)
                    {
                        OnCommentDetected(match);
                        pointer += match.Length;
                        lineNumber += match.Value.Count(c => c == '\n');
                    }
                }
            } while (pointer < src.Length && pointer != currentPass);
        }

        private void OnCommentDetected(Match match)
        {
            for (int i = 1; i < match.Groups.Count; i++)
            {
                var group = match.Groups[i];
                if (group.Success)
                {
                    var span = new Span(pointer, pointer + match.Length);
                    CommentDetected?.Invoke(this, new CommentEventArgs(group.Value, span));
                    break;
                }
            }
        }

        private Token NextToken()
        {
            foreach (var pat in patterns)
            {
                if (pat.TryMatch(src, pointer, out Token tok))
                {
                    pointer = tok.Span.End;
                    if (tok.Value != null)
                    {
                        lineNumber += tok.Value.Count(c => c == '\n');
                    }
                    return tok;
                }
            }

            var column = CurrentColumn();
            throw new UnrecognisedCharacterException(lineNumber + 1, column + 1, src[pointer]);
        }

        private int CurrentColumn()
        {
            var lastNewline = src.LastIndexOf('\n', pointer);
            return lastNewline < 0 ? pointer : pointer - lastNewline;
        }
    }

    /// <summary>
    /// The event arguments passed in when the <see cref="Lexer.CommentDetected"/>
    /// event is fired.
    /// </summary>
    public class CommentEventArgs : EventArgs
    {
        internal CommentEventArgs(string comment, Span span)
        {
            Comment = comment ?? throw new ArgumentNullException(nameof(comment));
            Span = span;
        }

        /// <summary>
        /// The comment's contents.
        /// </summary>
        public string Comment { get; }
        /// <summary>
        /// The location of the comment in the source text.
        /// </summary>
        public Span Span { get; }
    }
}
